/////////////////////////////////////////////////////////////////////////////////

// Original obtained from ShaderToy.com
// Adapted, trivialy, for VGHD by TheEmu.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

#define iGlobalTime u_Elapsed
#define iResolution u_WindowSize
#define iMouse AUTO_MOUSE

/////////////////////////////////////////////////////////////////////////////////

// Simple "Automatic Mouse". Simulates scanning the mouse over the full range of
// the screen with the X and Y scanning frequencies being different. TheEmu.

#define MOUSE_SPEED vec2(0.2,0.277777)
#define MOUSE_POS   vec2((1.0+cos(iGlobalTime*MOUSE_SPEED))*u_WindowSize/2.0)
#define MOUSE_PRESS vec2(0.0,0.0)
#define AUTO_MOUSE  vec4( MOUSE_POS, MOUSE_PRESS )

/////////////////////////////////////////////////////////////////////////////////

const float pi = 3.14159265359;
const vec3 ldir = vec3(0,0,1);
const float water = .7;

const float R_INNER = 1.0;
const float R = R_INNER * 1.04;

#define SEED 171.

//###### scatter
// credits to GLTracy

// math const
const float DEG_TO_RAD = pi / 180.0;
const float MAX = 10000.0;

// scatter const
const float K_R = 0.166;
const float K_M = 0.025;
const float E = 14.3; 						// light intensity
const vec3  C_R = vec3(0.3, 0.7, 1.0); 	// 1 / wavelength ^ 4
const float G_M = -0.85;					// Mie g

const float SCALE_H = 4.0 / ( R - R_INNER );
const float SCALE_L = 1.0 / ( R - R_INNER );

const int NUM_OUT_SCATTER = 10;
const float FNUM_OUT_SCATTER = 10.0;

const int NUM_IN_SCATTER = 10;
const float FNUM_IN_SCATTER = 10.0;

// ray intersects sphere
// e = -b +/- sqrt( b^2 - c )
vec2 ray_vs_sphere( vec3 p, vec3 dir, float r )
{
	float b = dot( p, dir );
	float c = dot( p, p ) - r * r;
	
	float d = b * b - c;
	if ( d < 0.0 ) {
		return vec2( MAX, -MAX );
	}
	d = sqrt( d );
	
	return vec2( -b - d, -b + d );
}

// Mie
// g : ( -0.75, -0.999 )
//      3 * ( 1 - g^2 )               1 + c^2
// F = ----------------- * -------------------------------
//      2 * ( 2 + g^2 )     ( 1 + g^2 - 2 * g * c )^(3/2)
float phase_mie( float g, float c, float cc )
{
	float gg = g * g;
	
	float a = ( 1.0 - gg ) * ( 1.0 + cc );

	float b = 1.0 + gg - 2.0 * g * c;
	b *= sqrt( b );
	b *= 2.0 + gg;	
	
	return 1.5 * a / b;
}

// Reyleigh
// g : 0
// F = 3/4 * ( 1 + c^2 )
float phase_reyleigh( float cc )
{
	return 0.75 * ( 1.0 + cc );
}

float density( vec3 p )
{
	return exp( -( length( p ) - R_INNER ) * SCALE_H );
}

float optic( vec3 p, vec3 q )
{
	vec3 step = ( q - p ) / FNUM_OUT_SCATTER;
	vec3 v = p + step * 0.5;
	
	float sum = 0.0;
	for ( int i = 0; i < NUM_OUT_SCATTER; i++ )
	{
		sum += density( v );
		v += step;
	}
	sum *= length( step ) * SCALE_L;
	
	return sum;
}

vec3 in_scatter( vec3 o, vec3 dir, vec2 e, vec3 l )
{
	float len = ( e.y - e.x ) / FNUM_IN_SCATTER;
	vec3 step = dir * len;
	vec3 p = o + dir * e.x;
	vec3 v = p + dir * ( len * 0.5 );

	vec3 sum = vec3( 0.0 );
	for ( int i = 0; i < NUM_IN_SCATTER; i++ )
	{
		vec2 f = ray_vs_sphere( v, l, R );
		vec3 u = v + l * f.y;
		
		float n = ( optic( p, v ) + optic( v, u ) ) * ( pi * 4.0 );
		
		sum += density( v ) * exp( -n * ( K_R * C_R + K_M ) );

		v += step;
	}
	sum *= len * SCALE_L;
	
	float c  = dot( dir, -l );
	float cc = c * c;
	
	return sum * ( K_R * C_R * phase_reyleigh( cc ) + K_M * phase_mie( G_M, c, cc ) ) * E;
}

vec3 scatter(vec3 ro, vec3 rd)
{
	vec2 e = ray_vs_sphere(ro, rd, R);
	if ( e.x > e.y )
		return vec3(0);
		
	vec2 f = ray_vs_sphere(ro, rd, R_INNER );
	e.y = min( e.y, f.x );
	
	return in_scatter( ro, rd, e, ldir );
}
//###### end scatter

//###### noise
// credits to iq for this noise algorithm

mat3 m = mat3( 0.00,  0.80,  0.60,
              -0.80,  0.36, -0.48,
              -0.60, -0.48,  0.64 );

float hash( float n )
{
    return fract(sin(n)*43758.5453);
}

float noise( in vec3 x )
{
    vec3 p = floor(x);
    vec3 f = fract(x);

    f = f*f*(3.0-2.0*f);

    float n = p.x + p.y*57.0 + 113.0*p.z;

    float res = mix(mix(mix( hash(n+  0.0), hash(n+  1.0),f.x),
                        mix( hash(n+ 57.0), hash(n+ 58.0),f.x),f.y),
                    mix(mix( hash(n+113.0), hash(n+114.0),f.x),
                        mix( hash(n+170.0), hash(n+171.0),f.x),f.y),f.z);
    return res;
}

float fbm( vec3 p )
{
    float f;
    f  = 0.5000*noise( p ); p = m*p*2.02;
    f += 0.2500*noise( p ); p = m*p*2.03;
    f += 0.1250*noise( p ); p = m*p*2.01;
    f += 0.0625*noise( p );
    return f;
}

//###### end noise

float map(in float value, in float istart, in float istop, in float ostart, in float ostop)
{
	return clamp(ostart + (ostop - ostart) * ((value - istart) / (istop - istart)),ostop,ostart);
}

vec3 waterColor(float h)
{
    return mix(vec3(0,.29,0.85), vec3(0,0,.25), h);
}

vec3 terrainColor(float h)
{
    h *= 1.2;
    return h < .5 ? mix(vec3(.91, .92, .48), vec3(.39, .52, .04), h * 2.) : 
    	   mix(vec3(.39, .52, .04), vec3(1.), (h - .5) * 2.);
}
    
float round(float r)
{
	return r - mod(r, 0.04);
}

float terrain(vec3 p)
{
	return fbm(p * 10.)/5. + fbm(p + SEED) - (1./5.);
}

bool isphere(vec3 p1, vec3 p2, vec3 p3, float r, out vec3 hitp)
{
	vec3 d = p2 - p1;

	float a = dot(d, d);
	float b = 2.0 * dot(d, p1 - p3);
	float c = dot(p3, p3) + dot(p1, p1) - 2.0 * dot(p3, p1) - r*r;

	float test = b*b - 4.0*a*c;

	if (test >= 0.0)
	{
		float u = (-b - sqrt(test)) / (2.0 * a);
		hitp = p1 + u * (p2 - p1);
		return true;
	}
	return false;
}

mat3 rm(vec3 axis, float angle)
{
    axis = normalize(axis);
    float s = sin(angle);
    float c = cos(angle);
    float oc = 1.0 - c;
    
    return mat3(oc * axis.x * axis.x + c,           oc * axis.x * axis.y - axis.z * s,  oc * axis.z * axis.x + axis.y * s,
                oc * axis.x * axis.y + axis.z * s,  oc * axis.y * axis.y + c,           oc * axis.y * axis.z - axis.x * s,
                oc * axis.z * axis.x - axis.y * s,  oc * axis.y * axis.z + axis.x * s,  oc * axis.z * axis.z + c);
}

vec3 render(vec3 ro, vec3 rd)
{
	mat3 r = rm(vec3(0,1,0), iGlobalTime/10.);
	vec3 hit, hit2;
	
	if (isphere(ro, ro+rd, vec3(0), R_INNER, hit) && dot(rd, hit-ro) > 0.)
	{
		hit = normalize(hit);
		hit2 = hit * r;
		float h = terrain(hit * r);
		vec3 c;
		if (h < (1.-water))
			c = terrainColor( round( abs( hit.y + h - (1. - water))));
		else
            c = waterColor( round(h/water));
		vec3 sc = scatter(ro, rd);
        
		return (sc + 0.7) * c * length(sc);
	}
	
	if (isphere(ro, ro+rd, vec3(0), R, hit) && dot(rd, hit-ro) > 0.)
		return scatter(ro, rd);
    return vec3(0);
}

vec3 getRay(vec3 pos, vec3 dir, vec3 up, vec2 offset)
{
	vec2 xy = (gl_FragCoord.xy + offset) / iResolution.xy - vec2(0.5,0.5);
	xy.y *= -iResolution.y / iResolution.x;
	
	vec3 eyed = normalize(dir);
    vec3 ud = normalize(cross(vec3(0.0,-1.0,0.0), eyed));
    vec3 vd = normalize(cross(eyed,ud));

	float fov = 67./180. * pi;
	float f = fov * length(xy);
	return normalize(normalize(xy.x*ud + xy.y*vd) + (1.0/tan(f))*eyed);
}

void main()
{
    float a = iMouse.x/iResolution.x*10.;
    float h = iMouse.y/iResolution.y - 0.5;
    
	vec3 pos = vec3(3.5*cos(a), h*5., 3.5*sin(a));
	vec3 dir = -normalize(pos);
	vec3 up =  vec3(0, 1, 0);
	
	vec3 color = render(pos, getRay(pos, dir, up, vec2(0)));
	
	gl_FragColor = vec4(color, 1);
}